<?php
/**
 * User Settings repository (DB + cache).
 *
 * Provides a small repository for reading the latest WhatsApp Cloud API
 * connection row from the `nxtcc_user_settings` table.
 *
 * @package NXTCC
 */

defined( 'ABSPATH' ) || exit;

/**
 * Data access wrapper for the `nxtcc_user_settings` table.
 *
 * Encapsulates DB reads used for connection checks and applies a bounded object cache.
 */
final class NXTCC_User_Settings_Repo {

	/**
	 * Cache group for user settings lookups.
	 *
	 * @var string
	 */
	private const CACHE_GROUP = 'nxtcc_user_settings';

	/**
	 * Singleton instance.
	 *
	 * @var self|null
	 */
	private static $instance = null;

	/**
	 * Get singleton instance.
	 *
	 * @return self
	 */
	public static function instance(): self {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * Intentionally private to enforce singleton usage.
	 */
	private function __construct() {}

	/**
	 * Quote a table name safely for SQL (backticked identifier).
	 *
	 * @param string $table Table name.
	 * @return string Backticked table identifier.
	 */
	private static function quote_table_name( string $table ): string {
		$clean = preg_replace( '/[^A-Za-z0-9_]/', '', $table );
		if ( ! is_string( $clean ) || '' === $clean ) {
			$clean = 'nxtcc_invalid';
		}

		return '`' . $clean . '`';
	}

	/**
	 * Prepare SQL that contains a single table placeholder token.
	 *
	 * @param string $query     SQL containing a %s table placeholder.
	 * @param string $table_sql Backtick-quoted table name.
	 * @param mixed  ...$args   Additional placeholders.
	 * @return string Prepared SQL, or empty string on failure.
	 */
	private static function prepare_with_table_token( string $query, string $table_sql, ...$args ): string {
		global $wpdb;

		$prepare_args = array_merge( array( '__NXTCC_TABLE__' ), $args );
		$prepared     = call_user_func_array(
			array( $wpdb, 'prepare' ),
			array_merge( array( $query ), $prepare_args )
		);

		if ( ! is_string( $prepared ) || '' === $prepared ) {
			return '';
		}

		return str_replace(
			array( "'__NXTCC_TABLE__'", '__NXTCC_TABLE__' ),
			$table_sql,
			$prepared
		);
	}

	/**
	 * Fetch the latest connection row (limited to required columns).
	 *
	 * Uses object cache to reduce repeated reads across a single request burst.
	 *
	 * @return object|null
	 */
	public function get_latest_connection_row(): ?object {
		$cache_key = 'latest_connection_row';
		$cached    = wp_cache_get( $cache_key, self::CACHE_GROUP );

		if ( false !== $cached ) {
			return is_object( $cached ) ? $cached : null;
		}

		global $wpdb;

		$table_sql = self::quote_table_name( $wpdb->prefix . 'nxtcc_user_settings' );
		$prepared  = self::prepare_with_table_token(
			'SELECT app_id, access_token_ct, access_token_nonce, business_account_id, phone_number_id
			 FROM %s
			 ORDER BY id DESC
			 LIMIT %d',
			$table_sql,
			1
		);
		if ( '' === $prepared ) {
			wp_cache_set( $cache_key, null, self::CACHE_GROUP, 300 );
			return null;
		}

		$row = call_user_func( array( $wpdb, 'get_row' ), $prepared );

		// Cache briefly (literal TTL so cache sniffers can evaluate it).
		wp_cache_set( $cache_key, $row ? $row : null, self::CACHE_GROUP, 300 );

		return $row ? $row : null;
	}
}
