<?php
/**
 * Send message helpers (text & media).
 *
 * Provides functional wrappers to send WhatsApp Cloud API messages and store
 * message history rows. This file is functions-only to satisfy PHPCS rules.
 *
 * @package NXTCC
 */

defined( 'ABSPATH' ) || exit;

require_once NXTCC_PLUGIN_DIR . 'includes/class-nxtcc-db-sendmessage.php';
require_once NXTCC_PLUGIN_DIR . 'includes/class-nxtcc-send-dao.php';
require_once NXTCC_PLUGIN_DIR . 'includes/class-nxtcc-helpers.php';
require_once NXTCC_PLUGIN_DIR . 'includes/nxtcc-helpers-functions.php';
require_once NXTCC_PLUGIN_DIR . 'includes/class-nxtcc-dao.php';
require_once NXTCC_PLUGIN_DIR . 'includes/nxtcc-remote.php';

if ( class_exists( 'NXTCC_DAO' ) ) {
	NXTCC_DAO::init();
}

/**
 * Static knowledge of wp_nxtcc_message_history columns.
 *
 * Avoids schema inspection queries. Extend this map when schema changes.
 *
 * @return array<string, bool>
 */
function nxtcc_mh_columns(): array {
	static $cols = null;

	if ( null === $cols ) {
		$cols = array(
			'user_mailid'          => true,
			'business_account_id'  => true,
			'phone_number_id'      => true,
			'contact_id'           => true,
			'display_phone_number' => true,
			'message_content'      => true,
			'status'               => true,
			'meta_message_id'      => true,
			'status_timestamps'    => true,
			'last_error'           => true,
			'response_json'        => true,
			'created_at'           => true,
			'sent_at'              => true,
			'delivered_at'         => true,
			'read_at'              => true,
			'failed_at'            => true,
			'is_read'              => true,
			'reply_to_wamid'       => true,
			'reply_to_history_id'  => true,
		);
	}

	return $cols;
}

/**
 * Check if a message-history column exists in our static map.
 *
 * @param string $col Column name.
 * @return bool
 */
function nxtcc_mh_has_column( string $col ): bool {
	$cols = nxtcc_mh_columns();
	return isset( $cols[ $col ] );
}

/**
 * Get a column max-length used for safe clipping.
 *
 * @param string $col Column name.
 * @return int|null
 */
function nxtcc_mh_col_maxlen( string $col ): ?int {
	static $max = array(
		'reply_to_wamid'       => 191,
		'meta_message_id'      => 191,
		'display_phone_number' => 32,
	);

	return isset( $max[ $col ] ) ? (int) $max[ $col ] : null;
}

/**
 * Decrypt access token for a (user, business_account_id, phone_number_id) tuple.
 *
 * @param string $user_mailid         User email.
 * @param string $business_account_id Business account ID.
 * @param string $phone_number_id     Phone number ID.
 * @return string|\WP_Error
 */
function nxtcc_get_decrypted_token( string $user_mailid, string $business_account_id, string $phone_number_id ) {
	$row = NXTCC_Send_DAO::get_settings_row( $user_mailid, $business_account_id, $phone_number_id );

	if ( ! $row ) {
		return new WP_Error( 'nxtcc_token_missing', 'Access token not found.' );
	}

	$token = nxtcc_crypto_decrypt(
		isset( $row->access_token_ct ) ? (string) $row->access_token_ct : null,
		isset( $row->access_token_nonce ) ? $row->access_token_nonce : null
	);

	if ( is_wp_error( $token ) || ! is_string( $token ) || '' === $token ) {
		return new WP_Error( 'nxtcc_token_decrypt_failed', 'Access token decryption failed.' );
	}

	return $token;
}

/**
 * Normalize recipient number into digits-only (E.164 digits).
 *
 * @param string $cc  Country code digits.
 * @param string $num Phone number digits.
 * @return string
 */
function nxtcc_normalize_recipient( string $cc, string $num ): string {
	$raw    = trim( $cc . $num );
	$digits = preg_replace( '/\D+/', '', $raw );

	if ( ! is_string( $digits ) ) {
		$digits = '';
	}

	$max = nxtcc_mh_col_maxlen( 'display_phone_number' );
	if ( null !== $max && strlen( $digits ) > $max ) {
		$digits = substr( $digits, 0, $max );
	}

	return $digits;
}

/**
 * Clip Meta message IDs if the schema has a limit.
 *
 * @param string|null $id Meta message ID.
 * @return string|null
 */
function nxtcc_clip_meta_message_id( ?string $id ): ?string {
	if ( null === $id || '' === $id ) {
		return null;
	}

	$max = nxtcc_mh_col_maxlen( 'meta_message_id' );
	if ( null !== $max && strlen( $id ) > $max ) {
		return substr( $id, 0, $max );
	}

	return $id;
}

/**
 * Validate/normalize a WhatsApp message "kind".
 *
 * @param string $kind Input kind.
 * @return string Normalized kind.
 */
function nxtcc_normalize_kind( string $kind ): string {
	$kind = strtolower( sanitize_text_field( $kind ) );

	$allowed = array( 'image', 'video', 'audio', 'document', 'sticker' );
	if ( in_array( $kind, $allowed, true ) ) {
		return $kind;
	}

	return 'document';
}

/**
 * Validate and clip reply-to message id (WAMID / Graph id).
 *
 * @param string $wamid Input id.
 * @return string Cleaned id (may be empty).
 */
function nxtcc_normalize_reply_wamid( string $wamid ): string {
	$wamid = sanitize_text_field( $wamid );
	$wamid = trim( $wamid );

	if ( '' === $wamid ) {
		return '';
	}

	// Allow common safe characters; drop anything suspicious.
	// (WAMID/Graph ids are typically URL-safe-ish strings).
	if ( 1 !== preg_match( '/^[a-zA-Z0-9._:-]{1,512}$/', $wamid ) ) {
		return '';
	}

	$max = nxtcc_mh_col_maxlen( 'reply_to_wamid' );
	if ( null !== $max && strlen( $wamid ) > $max ) {
		$wamid = substr( $wamid, 0, $max );
	}

	return $wamid;
}

/**
 * Safe JSON encode helper (returns empty string on failure).
 *
 * @param mixed $value Value.
 * @return string
 */
function nxtcc_json_encode_safe( $value ): string {
	$json = wp_json_encode( $value );
	return is_string( $json ) ? $json : '';
}

/**
 * Normalize media caption for WhatsApp API payloads.
 *
 * WhatsApp Cloud API media captions are limited to 1024 chars.
 *
 * @param string $caption Raw caption.
 * @return string Sanitized caption (possibly truncated).
 */
function nxtcc_normalize_media_caption( string $caption ): string {
	$caption = sanitize_textarea_field( $caption );
	$caption = trim( $caption );

	if ( '' === $caption ) {
		return '';
	}

	if ( function_exists( 'mb_strlen' ) && function_exists( 'mb_substr' ) ) {
		if ( mb_strlen( $caption, 'UTF-8' ) > 1024 ) {
			$caption = mb_substr( $caption, 0, 1024, 'UTF-8' );
		}
	} elseif ( strlen( $caption ) > 1024 ) {
		$caption = substr( $caption, 0, 1024 );
	}

	return $caption;
}

/**
 * Send a text message via WhatsApp Cloud API and insert history.
 *
 * Required args:
 * - user_mailid
 * - business_account_id
 * - phone_number_id
 * - contact_id
 * - message_content
 *
 * Optional args:
 * - reply_to_message_id
 * - reply_to_history_id
 *
 * @param array $args Input args.
 * @return array<string, mixed>
 */
function nxtcc_send_message_immediately( array $args ): array {
	global $wpdb;

	$required = array( 'user_mailid', 'business_account_id', 'phone_number_id', 'contact_id', 'message_content' );
	foreach ( $required as $key ) {
		if ( ! isset( $args[ $key ] ) || '' === (string) $args[ $key ] ) {
			return array(
				'success' => false,
				'error'   => 'Missing param: ' . $key,
			);
		}
	}

	$user_mailid         = sanitize_email( (string) $args['user_mailid'] );
	$business_account_id = sanitize_text_field( (string) $args['business_account_id'] );
	$phone_number_id     = sanitize_text_field( (string) $args['phone_number_id'] );
	$contact_id          = (int) $args['contact_id'];

	$message_content = sanitize_textarea_field( (string) $args['message_content'] );
	$message_content = trim( $message_content );

	$reply_to_message_id = isset( $args['reply_to_message_id'] ) ? nxtcc_normalize_reply_wamid( (string) $args['reply_to_message_id'] ) : '';
	$reply_to_history_id = isset( $args['reply_to_history_id'] ) ? (int) $args['reply_to_history_id'] : 0;

	if ( ! is_user_logged_in() ) {
		return array(
			'success' => false,
			'error'   => 'Unauthorized',
		);
	}

	$user = wp_get_current_user();
	if ( ! is_object( $user ) || (string) $user->user_email !== $user_mailid ) {
		return array(
			'success' => false,
			'error'   => 'Unauthorized',
		);
	}

	if ( '' === $user_mailid || '' === $business_account_id || '' === $phone_number_id || 0 === $contact_id ) {
		return array(
			'success' => false,
			'error'   => 'Invalid input',
		);
	}

	if ( '' === $message_content ) {
		return array(
			'success' => false,
			'error'   => 'Empty message',
		);
	}

	$contact = NXTCC_Send_DAO::get_contact_row( $contact_id, $user_mailid );
	if ( ! $contact ) {
		return array(
			'success' => false,
			'error'   => 'Invalid contact',
		);
	}

	$recipient = nxtcc_normalize_recipient(
		isset( $contact->country_code ) ? (string) $contact->country_code : '',
		isset( $contact->phone_number ) ? (string) $contact->phone_number : ''
	);

	if ( '' === $recipient ) {
		return array(
			'success' => false,
			'error'   => 'Invalid recipient number',
		);
	}

	$token = nxtcc_get_decrypted_token( $user_mailid, $business_account_id, $phone_number_id );
	if ( is_wp_error( $token ) ) {
		return array(
			'success' => false,
			'error'   => $token->get_error_message(),
		);
	}

	$url = 'https://graph.facebook.com/v19.0/' . rawurlencode( $phone_number_id ) . '/messages';

	$payload = array(
		'messaging_product' => 'whatsapp',
		'to'                => $recipient,
		'type'              => 'text',
		'text'              => array( 'body' => $message_content ),
	);

	if ( '' !== $reply_to_message_id ) {
		$payload['context'] = array( 'message_id' => $reply_to_message_id );
	}

	$body_json = nxtcc_json_encode_safe( $payload );
	if ( '' === $body_json ) {
		return array(
			'success' => false,
			'error'   => 'json_encode_failed',
		);
	}

	$response = nxtcc_safe_remote_post(
		$url,
		array(
			'headers' => array(
				'Authorization' => 'Bearer ' . $token,
				'Content-Type'  => 'application/json',
			),
			'body'    => $body_json,
			'timeout' => 3,
		)
	);

	$timestamp = current_time( 'mysql', 1 );
	$meta_raw  = is_wp_error( $response ) ? $response->get_error_message() : wp_remote_retrieve_body( $response );
	$parsed    = array();

	if ( is_string( $meta_raw ) ) {
		$decoded = json_decode( $meta_raw, true );
		if ( is_array( $decoded ) ) {
			$parsed = $decoded;
		}
	}

	$meta_status = 'failed';
	$meta_msg_id = null;
	$meta_err    = null;
	$timestamps  = array();

	if ( ! is_wp_error( $response ) && ! empty( $parsed['messages'][0]['id'] ) ) {
		$meta_status        = 'sent';
		$meta_msg_id        = nxtcc_clip_meta_message_id( (string) $parsed['messages'][0]['id'] );
		$timestamps['sent'] = $timestamp;
	} elseif ( ! is_wp_error( $response ) && ! empty( $parsed['error'] ) ) {
		$meta_err = isset( $parsed['error']['message'] ) ? (string) $parsed['error']['message'] : 'Graph error';
	} else {
		$meta_err = is_wp_error( $response ) ? $response->get_error_message() : null;
	}

	$row = array(
		'user_mailid'          => $user_mailid,
		'business_account_id'  => $business_account_id,
		'phone_number_id'      => $phone_number_id,
		'contact_id'           => $contact_id,
		'display_phone_number' => $recipient,
		'message_content'      => $message_content,
		'status'               => $meta_status,
		'meta_message_id'      => $meta_msg_id,
		'status_timestamps'    => nxtcc_json_encode_safe( $timestamps ),
		'last_error'           => $meta_err,
		'response_json'        => is_string( $meta_raw ) ? (string) $meta_raw : nxtcc_json_encode_safe( $parsed ),
		'created_at'           => $timestamp,
		'sent_at'              => isset( $timestamps['sent'] ) ? $timestamps['sent'] : null,
		'delivered_at'         => null,
		'read_at'              => null,
		'failed_at'            => ( 'failed' === $meta_status ) ? $timestamp : null,
		'is_read'              => 1,
	);

	if ( 0 === $reply_to_history_id && '' !== $reply_to_message_id && nxtcc_mh_has_column( 'reply_to_history_id' ) ) {
		$reply_to_history_id = NXTCC_Send_DAO::get_history_id_by_wamid( $reply_to_message_id );
	}

	if ( 0 !== $reply_to_history_id && nxtcc_mh_has_column( 'reply_to_history_id' ) ) {
		$row['reply_to_history_id'] = $reply_to_history_id;
	}

	if ( '' !== $reply_to_message_id && nxtcc_mh_has_column( 'reply_to_wamid' ) ) {
		$row['reply_to_wamid'] = $reply_to_message_id;
	}

	$ok = NXTCC_Send_DAO::insert_history( $row );

	if ( ! $ok ) {
		$db_err = ( is_string( $wpdb->last_error ) && '' !== $wpdb->last_error ) ? $wpdb->last_error : 'insert_failed';

		return array(
			'success'       => false,
			'status'        => $meta_status,
			'error'         => 'db_insert_failed',
			'db_error'      => $db_err,
			'meta_response' => $parsed,
		);
	}

	return array(
		'success'             => ( 'sent' === $meta_status ),
		'status'              => $meta_status,
		'meta_response'       => $parsed,
		'insert_id'           => (int) $wpdb->insert_id,
		'meta_message_id'     => $meta_msg_id,
		'reply_to_message_id' => ( '' !== $reply_to_message_id ) ? $reply_to_message_id : null,
	);
}

/**
 * Send a media message (link) via WhatsApp Cloud API and insert history.
 *
 * Required args:
 * - user_mailid
 * - business_account_id
 * - phone_number_id
 * - contact_id
 * - kind (image|video|audio|document|sticker)
 * - link
 *
 * Optional args:
 * - caption
 * - filename
 * - mime_type
 * - reply_to_message_id
 * - reply_to_history_id
 *
 * @param array $args Input args.
 * @return array<string, mixed>
 */
function nxtcc_send_media_link_immediately( array $args ): array {
	global $wpdb;

	$required = array( 'user_mailid', 'business_account_id', 'phone_number_id', 'contact_id', 'kind', 'link' );
	foreach ( $required as $key ) {
		if ( ! isset( $args[ $key ] ) || '' === (string) $args[ $key ] ) {
			return array(
				'success' => false,
				'error'   => 'Missing param: ' . $key,
			);
		}
	}

	$user_mailid         = sanitize_email( (string) $args['user_mailid'] );
	$business_account_id = sanitize_text_field( (string) $args['business_account_id'] );
	$phone_number_id     = sanitize_text_field( (string) $args['phone_number_id'] );
	$contact_id          = (int) $args['contact_id'];

	$kind_in             = nxtcc_normalize_kind( (string) $args['kind'] );
	$link                = esc_url_raw( (string) $args['link'] );
	$caption             = isset( $args['caption'] ) ? nxtcc_normalize_media_caption( (string) $args['caption'] ) : '';
	$filename            = isset( $args['filename'] ) ? sanitize_file_name( (string) $args['filename'] ) : '';
	$mime_type           = isset( $args['mime_type'] ) ? sanitize_text_field( (string) $args['mime_type'] ) : '';
	$reply_to_message_id = isset( $args['reply_to_message_id'] ) ? nxtcc_normalize_reply_wamid( (string) $args['reply_to_message_id'] ) : '';
	$reply_to_history_id = isset( $args['reply_to_history_id'] ) ? (int) $args['reply_to_history_id'] : 0;

	if ( ! is_user_logged_in() ) {
		return array(
			'success' => false,
			'error'   => 'Unauthorized',
		);
	}

	$user = wp_get_current_user();
	if ( ! is_object( $user ) || (string) $user->user_email !== $user_mailid ) {
		return array(
			'success' => false,
			'error'   => 'Unauthorized',
		);
	}

	if ( '' === $user_mailid || '' === $business_account_id || '' === $phone_number_id || 0 === $contact_id || '' === $link ) {
		return array(
			'success' => false,
			'error'   => 'Invalid input',
		);
	}

	$contact = NXTCC_Send_DAO::get_contact_row( $contact_id, $user_mailid );
	if ( ! $contact ) {
		return array(
			'success' => false,
			'error'   => 'Invalid contact',
		);
	}

	$recipient = nxtcc_normalize_recipient(
		isset( $contact->country_code ) ? (string) $contact->country_code : '',
		isset( $contact->phone_number ) ? (string) $contact->phone_number : ''
	);

	if ( '' === $recipient ) {
		return array(
			'success' => false,
			'error'   => 'Invalid recipient number',
		);
	}

	$token = nxtcc_get_decrypted_token( $user_mailid, $business_account_id, $phone_number_id );
	if ( is_wp_error( $token ) ) {
		return array(
			'success' => false,
			'error'   => $token->get_error_message(),
		);
	}

	$url = 'https://graph.facebook.com/v19.0/' . rawurlencode( $phone_number_id ) . '/messages';

	// Keep your existing behavior: sticker sends as image link (not an actual WA "sticker" type).
	$kind_for_api = ( 'sticker' === $kind_in ) ? 'image' : $kind_in;

	$payload = array(
		'messaging_product' => 'whatsapp',
		'to'                => $recipient,
	);

	if ( 'image' === $kind_for_api ) {
		$payload['type']  = 'image';
		$payload['image'] = array( 'link' => $link );
		if ( '' !== $caption ) {
			$payload['image']['caption'] = $caption;
		}
	} elseif ( 'video' === $kind_for_api ) {
		$payload['type']  = 'video';
		$payload['video'] = array( 'link' => $link );
		if ( '' !== $caption ) {
			$payload['video']['caption'] = $caption;
		}
	} elseif ( 'audio' === $kind_for_api ) {
		$payload['type']  = 'audio';
		$payload['audio'] = array( 'link' => $link );
	} else {
		$payload['type']     = 'document';
		$payload['document'] = array( 'link' => $link );
		if ( '' !== $caption ) {
			$payload['document']['caption'] = $caption;
		}
		if ( '' !== $filename ) {
			$payload['document']['filename'] = $filename;
		}
	}

	if ( '' !== $reply_to_message_id ) {
		$payload['context'] = array( 'message_id' => $reply_to_message_id );
	}

	$body_json = nxtcc_json_encode_safe( $payload );
	if ( '' === $body_json ) {
		return array(
			'success' => false,
			'error'   => 'json_encode_failed',
		);
	}

	$response = nxtcc_safe_remote_post(
		$url,
		array(
			'headers' => array(
				'Authorization' => 'Bearer ' . $token,
				'Content-Type'  => 'application/json',
			),
			'body'    => $body_json,
			'timeout' => 3,
		)
	);

	$timestamp = current_time( 'mysql', 1 );
	$meta_raw  = is_wp_error( $response ) ? $response->get_error_message() : wp_remote_retrieve_body( $response );
	$parsed    = array();

	if ( is_string( $meta_raw ) ) {
		$decoded = json_decode( $meta_raw, true );
		if ( is_array( $decoded ) ) {
			$parsed = $decoded;
		}
	}

	$meta_status = 'failed';
	$meta_msg_id = null;
	$timestamps  = array();

	if ( ! is_wp_error( $response ) && ! empty( $parsed['messages'][0]['id'] ) ) {
		$meta_status        = 'sent';
		$meta_msg_id        = nxtcc_clip_meta_message_id( (string) $parsed['messages'][0]['id'] );
		$timestamps['sent'] = $timestamp;
	}

	$normalized = array(
		'kind'      => $kind_in,
		'link'      => $link,
		'caption'   => ( '' !== $caption ) ? $caption : null,
		'filename'  => ( '' !== $filename ) ? $filename : null,
		'mime_type' => ( '' !== $mime_type ) ? $mime_type : null,
	);

	$row = array(
		'user_mailid'          => $user_mailid,
		'business_account_id'  => $business_account_id,
		'phone_number_id'      => $phone_number_id,
		'contact_id'           => $contact_id,
		'display_phone_number' => $recipient,
		'message_content'      => nxtcc_json_encode_safe( $normalized ),
		'status'               => $meta_status,
		'meta_message_id'      => $meta_msg_id,
		'status_timestamps'    => nxtcc_json_encode_safe( $timestamps ),
		'last_error'           => is_wp_error( $response ) ? $response->get_error_message() : null,
		'response_json'        => is_string( $meta_raw ) ? (string) $meta_raw : nxtcc_json_encode_safe( $parsed ),
		'created_at'           => $timestamp,
		'sent_at'              => isset( $timestamps['sent'] ) ? $timestamps['sent'] : null,
		'delivered_at'         => null,
		'read_at'              => null,
		'failed_at'            => ( 'failed' === $meta_status ) ? $timestamp : null,
		'is_read'              => 1,
	);

	if ( 0 === $reply_to_history_id && '' !== $reply_to_message_id && nxtcc_mh_has_column( 'reply_to_history_id' ) ) {
		$reply_to_history_id = NXTCC_Send_DAO::get_history_id_by_wamid( $reply_to_message_id );
	}

	if ( 0 !== $reply_to_history_id && nxtcc_mh_has_column( 'reply_to_history_id' ) ) {
		$row['reply_to_history_id'] = $reply_to_history_id;
	}

	if ( '' !== $reply_to_message_id && nxtcc_mh_has_column( 'reply_to_wamid' ) ) {
		$row['reply_to_wamid'] = $reply_to_message_id;
	}

	$ok = NXTCC_Send_DAO::insert_history( $row );

	if ( ! $ok ) {
		$db_err = ( is_string( $wpdb->last_error ) && '' !== $wpdb->last_error ) ? $wpdb->last_error : 'insert_failed';

		return array(
			'success'       => false,
			'status'        => $meta_status,
			'error'         => 'db_insert_failed',
			'db_error'      => $db_err,
			'meta_response' => $parsed,
		);
	}

	return array(
		'success'         => ( 'sent' === $meta_status ),
		'status'          => $meta_status,
		'meta_response'   => $parsed,
		'insert_id'       => (int) $wpdb->insert_id,
		'meta_message_id' => $meta_msg_id,
	);
}
