<?php
/**
 * Pages DAO for admin view templates.
 *
 * Centralizes data access used by files under /admin/pages/.
 *
 * Current scope:
 * - Fetch the latest WhatsApp connection settings row for a given admin user.
 *
 * Design goals:
 * - Keep view templates free of SQL.
 * - Use short object caching to limit repeated reads.
 * - Keep DB access centralized in one DAO.
 *
 * @package NXTCC
 */

defined( 'ABSPATH' ) || exit;

/**
 * Data access layer for admin pages.
 */
final class NXTCC_Pages_DAO {

	/**
	 * Cache group for admin page lookups.
	 */
	private const CACHE_GROUP = 'nxtcc_pages_dao';

	/**
	 * Quote a table identifier for SQL usage.
	 *
	 * @param string $table Table name.
	 * @return string Backtick-quoted table name.
	 */
	private static function quote_table_name( string $table ): string {
		return '`' . str_replace( '`', '', $table ) . '`';
	}

	/**
	 * Prepare SQL that contains a single table placeholder token.
	 *
	 * @param string $query     SQL query containing a %s table placeholder.
	 * @param string $table_sql Backtick-quoted table name.
	 * @param mixed  ...$args   Additional placeholders.
	 * @return string Prepared SQL, or empty string on failure.
	 */
	private static function prepare_with_table_token( string $query, string $table_sql, ...$args ): string {
		global $wpdb;

		$prepare_args = array_merge( array( '__NXTCC_TABLE__' ), $args );
		$prepared     = call_user_func_array(
			array( $wpdb, 'prepare' ),
			array_merge( array( $query ), $prepare_args )
		);
		if ( ! is_string( $prepared ) || '' === $prepared ) {
			return '';
		}

		return str_replace(
			array( "'__NXTCC_TABLE__'", '__NXTCC_TABLE__' ),
			$table_sql,
			$prepared
		);
	}

	/**
	 * Fetch latest settings row for a given admin email.
	 *
	 * Returns stdClass (similar to $wpdb->get_row()) so templates can read:
	 * $row->business_account_id, $row->phone_number_id, etc.
	 *
	 * @param string $user_mailid Admin email.
	 * @return stdClass|null Settings row or null.
	 */
	public static function get_latest_settings_row_for_user( string $user_mailid ) {
		global $wpdb;

		$user_mailid = sanitize_email( $user_mailid );
		if ( '' === $user_mailid ) {
			return null;
		}

		$cache_key = 'latest_settings_row_' . md5( strtolower( $user_mailid ) );

		$cached = wp_cache_get( $cache_key, self::CACHE_GROUP );
		if ( false !== $cached ) {
			return ( $cached instanceof stdClass ) ? $cached : null;
		}

		$table_sql = self::quote_table_name( $wpdb->prefix . 'nxtcc_user_settings' );
		$prepared  = self::prepare_with_table_token(
			'SELECT *
			 FROM %s
			 WHERE user_mailid = %s
			 ORDER BY id DESC
			 LIMIT 1',
			$table_sql,
			$user_mailid
		);
		if ( '' === $prepared ) {
			return null;
		}

		$row = call_user_func( array( $wpdb, 'get_row' ), $prepared );
		$row = ( $row instanceof stdClass ) ? $row : null;

		wp_cache_set( $cache_key, $row, self::CACHE_GROUP, 300 );

		return $row;
	}

	/**
	 * Flush cached settings row for a user.
	 *
	 * Call this after settings are updated.
	 *
	 * @param string $user_mailid Admin email.
	 * @return void
	 */
	public static function flush_latest_settings_row_cache( string $user_mailid ): void {
		$user_mailid = sanitize_email( $user_mailid );
		if ( '' === $user_mailid ) {
			return;
		}

		$cache_key = 'latest_settings_row_' . md5( strtolower( $user_mailid ) );

		wp_cache_delete( $cache_key, self::CACHE_GROUP );
	}
}
